<?php
/**
 * Plugin Name: PXS Cookies Pro (Polished v7.1.3 - Consent Fix)
 * Description: Wtyczka do cookies z obsługą Consent Mode v2 (PL/EN).
 * Version: 7.1.3
 * Author: PixelShark
 * Author URI: https://pixelshark.eu
 * Text Domain: pxs-cookies
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('PXS_Cookies_System')) {

    class PXS_Cookies_System {

        private $option_name = 'pxs_cookies_settings';

        public function __construct() {
            add_action('admin_menu', [$this, 'add_admin_menu']);
            add_action('admin_init', [$this, 'register_settings']);
            add_action('admin_init', [$this, 'handle_reset']);
            add_action('wp_head', [$this, 'render_head_scripts'], 0);
            add_action('wp_footer', [$this, 'render_frontend']);
        }

        private function get_defaults() {
            return [
                'lang' => 'pl',

                'heading_text_pl'     => 'Dbamy o Twoją prywatność',
                'body_text_pl'        => 'Używamy plików cookie, aby poprawić jakość przeglądania. Możesz zaakceptować wszystkie lub dostosować zgody.',
                'modal_heading_pl'    => 'Preferencje Ciasteczek',
                'modal_body_pl'       => 'Zarządzaj swoimi zgodami. Poniżej znajdziesz listę kategorii i używanych plików cookie.',
                'cat_necessary_pl'    => 'Niezbędne', 'desc_necessary_pl'   => 'Wymagane do działania strony.', 'list_necessary_pl'   => 'PHPSESSID, wp-settings-*',
                'cat_analytics_pl'    => 'Analityka', 'desc_analytics_pl'   => 'Anonimowe dane o ruchu.', 'list_analytics_pl'   => '_ga, _gid, _gat',
                'cat_marketing_pl'    => 'Marketing', 'desc_marketing_pl'   => 'Reklamy dopasowane do Ciebie.', 'list_marketing_pl'   => '_fbp, ads_prefs',
                'btn_accept_all_pl'   => 'Akceptuj wszystkie', 'btn_reject_all_pl'   => 'Odrzuć opcjonalne', 'btn_save_pl'         => 'Zapisz wybrane', 'btn_customize_pl'    => 'Dostosuj zgody',

                'heading_text_en'     => 'We value your privacy',
                'body_text_en'        => 'We use cookies to enhance your browsing experience. You can accept all or customise your preferences.',
                'modal_heading_en'    => 'Cookie Preferences',
                'modal_body_en'       => 'Manage your consent. Below is a list of categories and the cookies they use.',
                'cat_necessary_en'    => 'Necessary', 'desc_necessary_en'   => 'Required for the website to function.', 'list_necessary_en'   => 'PHPSESSID, wp-settings-*',
                'cat_analytics_en'    => 'Analytics', 'desc_analytics_en'   => 'Anonymous traffic data.', 'list_analytics_en'   => '_ga, _gid, _gat',
                'cat_marketing_en'    => 'Marketing', 'desc_marketing_en'   => 'Advertisements tailored to you.', 'list_marketing_en'   => '_fbp, ads_prefs',
                'btn_accept_all_en'   => 'Accept All', 'btn_reject_all_en'   => 'Reject Optional', 'btn_save_en'         => 'Save Selection', 'btn_customize_en'    => 'Customise Choices',

                'color_bg'          => '#ffffff', 'color_text'          => '#1e293b', 'color_primary'       => '#2563eb', 'color_secondary'     => '#64748b',
                'color_toggle_on'   => '#10b981', 'color_border'        => '#e2e8f0', 'color_trigger'       => '#2563eb', 'border_radius'       => '12',
            ];
        }

        private function get_options() {
            $saved = get_option($this->option_name, []);
            if (!is_array($saved)) {
                $saved = [];
            }
            return wp_parse_args($saved, $this->get_defaults());
        }

        private function get_text($opts, $key) {
            $lang = $opts['lang'] ?? 'pl';
            $key_lang = $key . '_' . $lang;
            return !empty($opts[$key_lang]) ? $opts[$key_lang] : $this->get_defaults()[$key_lang];
        }

        public function add_admin_menu() {
            add_menu_page('PXS Cookies', 'PXS Cookies', 'manage_options', 'pxs-cookies', [$this, 'render_admin_page'], 'dashicons-shield', 80);
        }

        public function register_settings() {
            register_setting('pxs_cookies_options', $this->option_name, [$this, 'sanitize_settings']);
        }

        public function sanitize_settings($input) {
            $current_options = get_option($this->option_name, []);
            if (!is_array($current_options)) {
                $current_options = [];
            }
            if (!$input || !is_array($input)) {
                return $current_options;
            }
            $merged = array_merge($current_options, $input);
            return $merged;
        }

        public function handle_reset() {
            if (isset($_POST['pxs_reset_action']) && current_user_can('manage_options')) {
                delete_option($this->option_name);
                add_settings_error($this->option_name, 'pxs_reset', 'Przywrócono ustawienia domyślne.', 'updated');
            }
        }

        public function render_admin_page() {
            $opts = $this->get_options();
            $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'content';
            $lang_label = ($opts['lang'] === 'en') ? 'EN' : 'PL';
            ?>
            <div class="wrap">
                <h1>PXS Cookies (Stable v7.1.3)</h1>
                <h2 class="nav-tab-wrapper">
                    <a href="?page=pxs-cookies&tab=content" class="nav-tab <?php echo $active_tab == 'content' ? 'nav-tab-active' : ''; ?>">1. Treść (<?php echo $lang_label; ?>)</a>
                    <a href="?page=pxs-cookies&tab=style" class="nav-tab <?php echo $active_tab == 'style' ? 'nav-tab-active' : ''; ?>">2. Wygląd</a>
                </h2>
                <form action="options.php" method="post">
                    <?php
                    settings_fields('pxs_cookies_options');
                    if ($active_tab == 'content') {
                        $this->render_content_tab($opts);
                    } elseif ($active_tab == 'style') {
                        $this->render_style_tab($opts);
                    }
                    submit_button('Zapisz zmiany');
                    ?>
                </form>
            </div>
            <?php
        }

        private function render_content_tab($opts) {
            $lang = $opts['lang'] ?? 'pl';
            echo '<div style="max-width:800px;">';
            echo '<h3>Ustawienia Języka</h3><table class="form-table">';
            echo '<tr><th>Edytuj wersję:</th><td>';
            echo '<select name="'.$this->option_name.'[lang]" onchange="this.form.submit()">';
            echo '<option value="pl" '.selected($lang, 'pl', false).'>Polski 🇵🇱</option>';
            echo '<option value="en" '.selected($lang, 'en', false).'>Angielski (UK) 🇬🇧</option>';
            echo '</select>';
            echo '<p class="description">Wybierz język, aby edytować jego teksty. Ustawienie to decyduje też, który język widzi użytkownik.</p></td></tr></table>';
            echo '<hr>';
            echo '<h3>Baner ('.$lang.')</h3><table class="form-table">';
            echo '<tr><th>Nagłówek:</th><td><input type="text" name="'.$this->option_name.'[heading_text_'.$lang.']" value="'.esc_attr($this->get_text($opts, 'heading_text')).'" class="large-text"></td></tr>';
            echo '<tr><th>Treść:</th><td><textarea name="'.$this->option_name.'[body_text_'.$lang.']" rows="2" class="large-text">'.esc_textarea($this->get_text($opts, 'body_text')).'</textarea></td></tr></table>';
            echo '<h3>Modal ('.$lang.')</h3><table class="form-table">';
            echo '<tr><th>Nagłówek:</th><td><input type="text" name="'.$this->option_name.'[modal_heading_'.$lang.']" value="'.esc_attr($this->get_text($opts, 'modal_heading')).'" class="large-text"></td></tr>';
            echo '<tr><th>Treść:</th><td><textarea name="'.$this->option_name.'[modal_body_'.$lang.']" rows="2" class="large-text">'.esc_textarea($this->get_text($opts, 'modal_body')).'</textarea></td></tr></table>';
            echo '<h3>Kategorie ('.$lang.')</h3><table class="form-table">';
            foreach (['necessary'=>'Niezbędne','analytics'=>'Analityka','marketing'=>'Marketing'] as $k=>$v) {
                $lbl = ($lang == 'en') ? (($k=='necessary')?'Necessary':(($k=='analytics')?'Analytics':'Marketing')) : $v;
                echo "<tr style='background:#f9f9f9;'><th>$lbl</th><td>
                <input type='text' name='{$this->option_name}[cat_{$k}_{$lang}]' value='".esc_attr($this->get_text($opts, "cat_$k"))."' class='regular-text' placeholder='Nazwa kategorii'><br>
                <input type='text' name='{$this->option_name}[desc_{$k}_{$lang}]' value='".esc_attr($this->get_text($opts, "desc_$k"))."' class='large-text' placeholder='Opis'><br>
                <small>Cookies:</small> <input type='text' name='{$this->option_name}[list_{$k}_{$lang}]' value='".esc_attr($this->get_text($opts, "list_$k"))."' class='large-text'></td></tr>";
            }
            echo '</table><h3>Przyciski ('.$lang.')</h3><table class="form-table">';
            $btns = ['btn_accept_all'=>'Akceptuj','btn_reject_all'=>'Odrzuć','btn_customize'=>'Dostosuj','btn_save'=>'Zapisz'];
            foreach ($btns as $k=>$v) {
                echo "<tr><th>$v:</th><td><input type='text' name='{$this->option_name}[{$k}_{$lang}]' value='".esc_attr($this->get_text($opts, $k))."' class='regular-text'></td></tr>";
            }
            echo '</table></div>';
        }

        private function render_style_tab($opts) {
            echo '<div style="margin-bottom:20px; padding:15px; border:1px solid #ddd; background:#fff;"><strong>Szybka zmiana motywu:</strong> <button type="button" class="button" onclick="pxsSetTheme(\'light\')">☀️ Jasny</button> <button type="button" class="button" onclick="pxsSetTheme(\'dark\')">🌙 Ciemny</button></div>';
            echo '<table class="form-table">';
            $colors = ['color_bg'=>'Tło','color_text'=>'Tekst','color_primary'=>'Akceptuj','color_secondary'=>'Odrzuć','color_toggle_on'=>'Switch ON','color_border'=>'Obramowanie','color_trigger'=>'Tło Ikonki'];
            foreach ($colors as $k=>$v) {
                echo "<tr><th>$v:</th><td><input type='color' id='c_$k' value='".esc_attr($opts[$k])."' style='cursor:pointer;'> <input type='text' name='{$this->option_name}[$k]' id='t_$k' value='".esc_attr($opts[$k])."' class='regular-text' oninput=\"document.getElementById('c_$k').value=this.value\"><script>document.getElementById('c_$k').addEventListener('input',function(){document.getElementById('t_$k').value=this.value;});</script></td></tr>";
            }
            echo '<tr><th>Zaokrąglenie (px):</th><td><input type="number" name="'.$this->option_name.'[border_radius]" value="'.esc_attr($opts['border_radius']).'" class="small-text"> px</td></tr></table>';
            echo '<script>function pxsSetTheme(m){if(!confirm("Zmienić kolory?"))return;const t=m==="light"?{"color_bg":"#ffffff","color_text":"#1e293b","color_primary":"#2563eb","color_secondary":"#64748b","color_toggle_on":"#10b981","color_border":"#e2e8f0","color_trigger":"#2563eb"}:{"color_bg":"#0f172a","color_text":"#f8fafc","color_primary":"#3b82f6","color_secondary":"#94a3b8","color_toggle_on":"#22c55e","color_border":"#334155","color_trigger":"#3b82f6"};for(const[k,v]of Object.entries(t)){document.getElementById("t_"+k).value=v;document.getElementById("c_"+k).value=v;}}</script>';
            echo '<hr><button type="submit" name="pxs_reset_action" value="1" class="button button-link-delete" onclick="return confirm(\'Jesteś pewien? To usunie wszystkie ustawienia.\');">Przywróć Domyślne</button>';
        }

        public function render_head_scripts() {
            ?>
            <script>
                window.dataLayer = window.dataLayer || [];
                window.gtag = window.gtag || function(){ window.dataLayer.push(arguments); };
                (function(){
                    var stored = null;
                    try {
                        stored = localStorage.getItem('pxs_consent_status');
                    } catch(e) {}
                    if (!stored) {
                        window.gtag('consent', 'default', {
                            ad_storage: 'denied',
                            ad_user_data: 'denied',
                            ad_personalization: 'denied',
                            analytics_storage: 'denied',
                            personalization_storage: 'denied',
                            functionality_storage: 'granted',
                            security_storage: 'granted',
                            wait_for_update: 500
                        });
                    } else {
                        try {
                            var parsed = JSON.parse(stored);
                            window.gtag('consent', 'default', parsed);
                        } catch(e) {
                            window.gtag('consent', 'default', {
                                ad_storage: 'denied',
                                ad_user_data: 'denied',
                                ad_personalization: 'denied',
                                analytics_storage: 'denied',
                                personalization_storage: 'denied',
                                functionality_storage: 'granted',
                                security_storage: 'granted',
                                wait_for_update: 500
                            });
                        }
                    }
                })();
            </script>
            <?php
        }

        public function render_frontend() {
            $opts = $this->get_options();
            $logo_url = plugin_dir_url(__FILE__) . 'pxs-cookie.webp';
            $t = [
                'head' => $this->get_text($opts, 'heading_text'),
                'body' => $this->get_text($opts, 'body_text'),
                'm_head' => $this->get_text($opts, 'modal_heading'),
                'm_body' => $this->get_text($opts, 'modal_body'),
                'accept' => $this->get_text($opts, 'btn_accept_all'),
                'reject' => $this->get_text($opts, 'btn_reject_all'),
                'cust' => $this->get_text($opts, 'btn_customize'),
                'save' => $this->get_text($opts, 'btn_save'),
            ];
            ?>
            <style>
                :root {
                    --pxs-bg: <?php echo esc_attr($opts['color_bg']); ?>;
                    --pxs-text: <?php echo esc_attr($opts['color_text']); ?>;
                    --pxs-prim: <?php echo esc_attr($opts['color_primary']); ?>;
                    --pxs-sec: <?php echo esc_attr($opts['color_secondary']); ?>;
                    --pxs-togg: <?php echo esc_attr($opts['color_toggle_on']); ?>;
                    --pxs-brd: <?php echo esc_attr($opts['color_border']); ?>;
                    --pxs-trig: <?php echo esc_attr($opts['color_trigger']); ?>;
                    --pxs-rad: <?php echo esc_attr($opts['border_radius']); ?>px;
                }
                #pxs-banner-wrap, #pxs-modal-wrap, #pxs-trigger-wrap { all: initial; font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif; }
                #pxs-banner *, #pxs-modal *, #pxs-trigger * { box-sizing: border-box !important; }
                #pxs-banner a, #pxs-modal a { color: inherit !important; text-decoration: none !important; box-shadow: none !important; border: none !important; background: none !important; }
                #pxs-banner button#pxs-open.pxs-link-cust { color: var(--pxs-text) !important; text-decoration: underline !important; }
                #pxs-banner h3, #pxs-modal h2, #pxs-modal h4 { margin: 0 0 10px 0 !important; padding: 0 !important; color: var(--pxs-text) !important; font-size: 16px !important; line-height: 1.3 !important; font-weight: 700 !important; }
                #pxs-banner p, #pxs-modal p { margin: 0 0 15px 0 !important; padding: 0 !important; color: var(--pxs-text) !important; font-size: 13px !important; line-height: 1.5 !important; }
                #pxs-banner button, #pxs-modal button { appearance: none !important; margin: 0 !important; line-height: 1 !important; }

                #pxs-banner {
                    position: fixed !important; bottom: 20px !important; left: 20px !important; width: 400px !important; max-width: 90% !important;
                    background: var(--pxs-bg) !important; color: var(--pxs-text) !important;
                    padding: 24px !important; border-radius: var(--pxs-rad) !important;
                    box-shadow: 0 10px 40px rgba(0,0,0,0.2) !important; border: 1px solid var(--pxs-brd) !important;
                    z-index: 999999 !important; display: none; backdrop-filter: blur(10px) !important;
                    background: <?php echo esc_attr($opts['color_bg']); ?>f2 !important;
                    animation: pxsUp 0.5s ease-out !important;
                }
                @keyframes pxsUp { from{opacity:0; transform:translateY(30px);} to{opacity:1; transform:translateY(0);} }
                .pxs-btns-row { display: flex !important; gap: 10px !important; margin-bottom: 15px !important; }
                .pxs-btn { flex: 1 !important; padding: 12px 15px !important; border-radius: 6px !important; font-size: 13px !important; font-weight: 600 !important; cursor: pointer !important; text-align: center !important; display: inline-block !important; border: none !important; }
                .pxs-btn:hover { filter: brightness(110%) !important; }
                .pxs-primary { background: var(--pxs-prim) !important; color: #fff !important; border: 1px solid var(--pxs-prim) !important; }
                .pxs-secondary { background: transparent !important; border: 1px solid var(--pxs-brd) !important; color: var(--pxs-text) !important; }
                .pxs-foot { display: flex !important; justify-content: space-between !important; align-items: center !important; width: 100% !important; margin-top: 10px !important; }
                .pxs-link-cust { background: none !important; border: none !important; cursor: pointer !important; font-size: 11px !important; opacity: 0.8 !important; padding: 0 !important; }
                .pxs-powered { opacity: 0.5 !important; font-size: 10px !important; text-align: right !important; }
                #pxs-banner .pxs-powered a { text-decoration: none !important; font-weight: bold !important; color: inherit !important; }
                #pxs-trigger { position: fixed !important; bottom: 20px !important; left: 20px !important; width: 55px !important; height: 55px !important; border-radius: 50% !important; background: var(--pxs-trig) !important; box-shadow: 0 4px 15px rgba(0,0,0,0.2) !important; z-index: 999990 !important; cursor: pointer !important; display: none; border: none !important; align-items: center !important; justify-content: center !important; }
                #pxs-trigger:hover { transform: scale(1.1) !important; }
                #pxs-trigger img { width: 80% !important; height: 80% !important; object-fit: contain !important; display: block !important; margin: 0 auto !important; }
                #pxs-modal-overlay { position: fixed !important; top:0 !important; left:0 !important; width:100% !important; height:100% !important; background:rgba(0,0,0,0.6) !important; z-index:1000000 !important; display:none; align-items:center !important; justify-content:center !important; backdrop-filter:blur(4px) !important; }
                #pxs-modal { background: var(--pxs-bg) !important; width: 550px !important; max-width: 95% !important; max-height: 90vh !important; border-radius: var(--pxs-rad) !important; display: flex !important; flex-direction: column !important; box-shadow: 0 25px 50px rgba(0,0,0,0.3) !important; overflow: hidden !important; color: var(--pxs-text) !important; }
                .pxs-m-head { padding: 20px !important; border-bottom: 1px solid var(--pxs-brd) !important; display: flex !important; justify-content: space-between !important; align-items: center !important; }
                .pxs-m-close { cursor: pointer !important; font-size: 24px !important; opacity: 0.5 !important; color: var(--pxs-text) !important; line-height: 1 !important; }
                .pxs-m-body { padding: 20px !important; overflow-y: auto !important; }
                .pxs-row { display: flex !important; justify-content: space-between !important; margin-bottom: 20px !important; border-bottom: 1px solid var(--pxs-brd) !important; padding-bottom: 20px !important; }
                .pxs-row:last-child { border: none !important; margin: 0 !important; padding: 0 !important; }
                .pxs-switch { position: relative !important; width: 40px !important; height: 22px !important; flex-shrink: 0 !important; display: inline-block !important; }
                .pxs-switch input { opacity: 0 !important; width: 0 !important; height: 0 !important; position: absolute !important; }
                .pxs-slider { position: absolute !important; top:0 !important; left:0 !important; right:0 !important; bottom:0 !important; background: #ccc !important; border-radius: 20px !important; transition: .3s !important; cursor: pointer !important; }
                .pxs-slider:before { content:"" !important; position: absolute !important; height: 16px !important; width: 16px !important; left: 3px !important; bottom: 3px !important; background: #fff !important; border-radius: 50% !important; transition: .3s !important; }
                input:checked + .pxs-slider { background: var(--pxs-togg) !important; }
                input:checked + .pxs-slider:before { transform: translateX(18px) !important; }
                input:disabled + .pxs-slider { opacity: 0.5 !important; cursor: not-allowed !important; }
                .pxs-m-foot { padding: 20px !important; border-top: 1px solid var(--pxs-brd) !important; background: rgba(0,0,0,0.02) !important; display: flex !important; flex-direction: column !important; align-items: center !important; }
                @media(max-width: 500px) { #pxs-banner { left: 10px !important; right: 10px !important; width: auto !important; } .pxs-btns-row { flex-direction: column !important; } }
            </style>

            <div id="pxs-trigger-wrap">
                <div id="pxs-trigger" title="<?php echo esc_attr($t['cust']); ?>">
                    <img src="<?php echo esc_url($logo_url); ?>" alt="Settings" onerror="this.style.display='none'; this.parentNode.innerHTML='🍪'; this.parentNode.style.fontSize='24px';this.parentNode.style.display='flex';">
                </div>
            </div>

            <div id="pxs-banner-wrap">
                <div id="pxs-banner">
                    <h3><?php echo esc_html($t['head']); ?></h3>
                    <p><?php echo esc_html($t['body']); ?></p>
                    <div class="pxs-btns-row">
                        <button id="pxs-accept" class="pxs-btn pxs-primary"><?php echo esc_html($t['accept']); ?></button>
                        <button id="pxs-reject" class="pxs-btn pxs-secondary"><?php echo esc_html($t['reject']); ?></button>
                    </div>
                    <div class="pxs-foot">
                        <button id="pxs-open" class="pxs-link-cust"><?php echo esc_html($t['cust']); ?></button>
                        <div class="pxs-powered">Powered by <a href="https://pixelshark.eu" target="_blank">PixelShark</a></div>
                    </div>
                </div>
            </div>

            <div id="pxs-modal-wrap">
                <div id="pxs-modal-overlay">
                    <div id="pxs-modal">
                        <div class="pxs-m-head">
                            <h2><?php echo esc_html($t['m_head']); ?></h2>
                            <span class="pxs-m-close" id="pxs-close">&times;</span>
                        </div>
                        <div class="pxs-m-body">
                            <p><?php echo esc_html($t['m_body']); ?></p>
                            <?php
                            $cats = [
                                'necessary'=>['title'=>$this->get_text($opts,'cat_necessary'), 'desc'=>$this->get_text($opts,'desc_necessary'), 'list'=>$this->get_text($opts,'list_necessary'), 'chk'=>true,'dis'=>true],
                                'analytics'=>['title'=>$this->get_text($opts,'cat_analytics'), 'desc'=>$this->get_text($opts,'desc_analytics'), 'list'=>$this->get_text($opts,'list_analytics'), 'id'=>'pxs-chk-ana'],
                                'marketing'=>['title'=>$this->get_text($opts,'cat_marketing'), 'desc'=>$this->get_text($opts,'desc_marketing'), 'list'=>$this->get_text($opts,'list_marketing'), 'id'=>'pxs-chk-mkt']
                            ];
                            foreach ($cats as $k=>$v) {
                                echo '<div class="pxs-row"><div><h4>'.esc_html($v['title']).'</h4><p>'.esc_html($v['desc']).'</p>';
                                if (!empty($v["list"])) {
                                    echo '<small style="opacity:0.6; font-family:monospace!important;">Cookies: '.esc_html($v["list"]).'</small>';
                                }
                                echo '</div><label class="pxs-switch"><input type="checkbox" '.(isset($v['id'])?'id="'.$v['id'].'"':'').' '.(isset($v['chk'])?'checked':'').' '.(isset($v['dis'])?'disabled':'').'><span class="pxs-slider"></span></label></div>';
                            }
                            ?>
                        </div>
                        <div class="pxs-m-foot">
                            <div class="pxs-btns-row" style="width:100%;">
                                <button id="pxs-save" class="pxs-btn pxs-primary"><?php echo esc_html($t['save']); ?></button>
                                <button id="pxs-accept-m" class="pxs-btn pxs-secondary"><?php echo esc_html($t['accept']); ?></button>
                            </div>
                            <div class="pxs-powered" style="width:100%; text-align:center!important;">Powered by <a href="https://pixelshark.eu" target="_blank">PixelShark</a></div>
                        </div>
                    </div>
                </div>
            </div>

            <script>
            document.addEventListener("DOMContentLoaded", function() {
                var els = {
                    banner: document.getElementById('pxs-banner'),
                    trigger: document.getElementById('pxs-trigger'),
                    overlay: document.getElementById('pxs-modal-overlay'),
                    close: document.getElementById('pxs-close'),
                    open: document.getElementById('pxs-open'),
                    accept: document.getElementById('pxs-accept'),
                    reject: document.getElementById('pxs-reject'),
                    save: document.getElementById('pxs-save'),
                    acceptM: document.getElementById('pxs-accept-m'),
                    chkA: document.getElementById('pxs-chk-ana'),
                    chkM: document.getElementById('pxs-chk-mkt')
                };
                function init() {
                    var c = null;
                    try {
                        c = localStorage.getItem('pxs_consent_status');
                    } catch(e) {}
                    if (!c) {
                        els.banner.style.display = 'block';
                        els.trigger.style.display = 'none';
                    } else {
                        els.banner.style.display = 'none';
                        els.trigger.style.display = 'flex';
                        try {
                            var p = JSON.parse(c);
                            els.chkA.checked = (p.analytics_storage === 'granted');
                            els.chkM.checked = (p.ad_storage === 'granted');
                        } catch(e) {}
                    }
                }
                function save(ana, mkt) {
                    var s = {
                        ad_storage: mkt ? 'granted' : 'denied',
                        ad_user_data: mkt ? 'granted' : 'denied',
                        ad_personalization: mkt ? 'granted' : 'denied',
                        analytics_storage: ana ? 'granted' : 'denied',
                        functionality_storage: 'granted',
                        security_storage: 'granted',
                        personalization_storage: mkt ? 'granted' : 'denied'
                    };
                    try {
                        localStorage.setItem('pxs_consent_status', JSON.stringify(s));
                    } catch(e) {}
                    if (typeof gtag === 'function') {
                        gtag('consent', 'update', s);
                        if (typeof dataLayer !== 'undefined' && dataLayer && typeof dataLayer.push === 'function') {
                            dataLayer.push({event: 'pxs_consent_update'});
                        }
                    }
                    els.banner.style.display = 'none';
                    els.overlay.style.display = 'none';
                    els.trigger.style.display = 'flex';
                    els.chkA.checked = ana;
                    els.chkM.checked = mkt;
                }
                els.accept.onclick = function(){ save(true, true); };
                els.acceptM.onclick = function(){ save(true, true); };
                els.reject.onclick = function(){ save(false, false); };
                els.save.onclick = function(){ save(els.chkA.checked, els.chkM.checked); };
                els.open.onclick = function(){ els.banner.style.display = 'none'; els.overlay.style.display = 'flex'; };
                els.trigger.onclick = function(){ els.trigger.style.display = 'none'; els.overlay.style.display = 'flex'; };
                els.close.onclick = function(){
                    els.overlay.style.display = 'none';
                    var c = null;
                    try {
                        c = localStorage.getItem('pxs_consent_status');
                    } catch(e) {}
                    if (!c) {
                        els.banner.style.display = 'block';
                    } else {
                        els.trigger.style.display = 'flex';
                    }
                };
                init();
            });
            </script>
            <?php
        }
    }

    new PXS_Cookies_System();
}

/* Crafted with ❤️ by PixelShark.eu // */
